/*
=============================================================================
Module Information
------------------
Name:			cnt_array.cpp
Author:			Rich Whitehouse
Description:	container - array
=============================================================================
*/

#include "cnt_arraylist.h"
#include "pluginshare.h"

typedef struct cntArray_s
{
	unsigned char		*elements;
	int					numElements;
	int					maxElements;
	int					elementSize;
	bool				exponential;

	void				*(*mem_malloc)(size_t size);
	void				*(*mem_realloc)(void *ptr, size_t newSize);
	void				(*mem_free)(void *ptr);
} cntArray_t;

//alloc
cntArray_t *Array_Alloc(int elementSize, int initialNum)
{
	assert(initialNum > 0);
	cntArray_t *ar = (cntArray_t *)malloc(sizeof(cntArray_t));
	memset(ar, 0, sizeof(cntArray_t));

	ar->mem_malloc = malloc;
	ar->mem_realloc = realloc;
	ar->mem_free = free;

	ar->elementSize = elementSize;
	ar->maxElements = initialNum;
	ar->elements = (unsigned char *)ar->mem_malloc(ar->maxElements*ar->elementSize);
	if (!ar->elements)
	{
		assert(0);
		return NULL;
	}
	memset(ar->elements, 0, ar->maxElements*ar->elementSize);

	return ar;
}

//free
void Array_Free(cntArray_t *ar)
{
	if (ar->elements)
	{
		ar->mem_free(ar->elements);
	}
	ar->mem_free(ar);
}

//set growth type
void Array_SetGrowth(cntArray_t *ar, bool exponential)
{
	ar->exponential = exponential;
}

//qsort it
void Array_QSort(cntArray_t *ar, int (__cdecl * compareFunc)(const void *a, const void *b))
{
	if (ar->numElements <= 0)
	{
		return;
	}
	qsort(ar->elements, ar->numElements, ar->elementSize, compareFunc);
}

//resize
static void Array_Resize(cntArray_t *ar, int num)
{
	if (ar->maxElements == num)
	{
		return;
	}

	unsigned char *newElements = (unsigned char *)ar->mem_malloc(num*ar->elementSize);
	assert(newElements);
	if (!newElements)
	{
		assert(0);
		return;
	}
	int copyElements = (num > ar->maxElements) ? ar->maxElements : num;
	memcpy(newElements, ar->elements, copyElements*ar->elementSize);
	if (num > ar->maxElements)
	{
		memset(newElements+(ar->maxElements*ar->elementSize), 0, (num-ar->maxElements)*ar->elementSize);
	}
	ar->maxElements = num;
	ar->mem_free(ar->elements);
	ar->elements = newElements;
}

//see if the array needs to grow
static void Array_CheckGrowth(cntArray_t *ar, int maxIndex)
{
	if (maxIndex <= ar->maxElements)
	{
		return;
	}

	if (ar->exponential)
	{
		int newMax = ar->maxElements;
		while (newMax < maxIndex)
		{
			newMax *= 2;
		}
		Array_Resize(ar, newMax);
	}
	else
	{
		Array_Resize(ar, maxIndex);
	}
}

//get element
void *Array_GetElement(cntArray_t *ar, int index)
{
	if (index < 0 || index >= ar->numElements)
	{
		return NULL;
	}
	return ar->elements+(index*ar->elementSize);
}

//get element
void *Array_GetElementGrow(cntArray_t *ar, int index)
{
	if (index < 0)
	{
		return NULL;
	}

	if (index >= ar->numElements)
	{
		unsigned char *ele = (unsigned char *)_alloca(ar->elementSize);
		memset(ele, 0, ar->elementSize);
		Array_Insert(ar, ele, index);
	}
	return ar->elements+(index*ar->elementSize);
}

//append
void Array_Append(cntArray_t *ar, void *element)
{
	Array_CheckGrowth(ar, ar->numElements+1);
	memcpy(ar->elements+(ar->numElements*ar->elementSize), element, ar->elementSize);
	ar->numElements++;
}

//remove last
void Array_RemoveLast(cntArray_t *ar)
{
	if (ar->numElements > 0)
	{
		ar->numElements--;
	}
}

//insert
void Array_Insert(cntArray_t *ar, void *element, int index)
{
	assert(index >= 0);
	Array_CheckGrowth(ar, index+1);
	memcpy(ar->elements+(index*ar->elementSize), element, ar->elementSize);
	if (ar->numElements <= index)
	{
		ar->numElements = index+1;
	}
}

//remove
void Array_Remove(cntArray_t *ar, int index)
{
	assert(index >= 0);
	if (index >= ar->numElements)
	{
		return;
	}

	if (index < ar->numElements-1)
	{
		memcpy(ar->elements+(index*ar->elementSize), ar->elements+((index+1)*ar->elementSize),
			(ar->numElements-index-1)*ar->elementSize);
	}
	ar->numElements--;
}

//get count
int Array_GetCount(cntArray_t *ar)
{
	return ar->numElements;
}

//reset without clearing/de-allocating
void Array_Reset(cntArray_t *ar)
{
	ar->numElements = 0;
}

//size down to the actual number of elements used
void Array_Tighten(cntArray_t *ar)
{
	if (ar->numElements >= ar->maxElements)
	{
		return;
	}
	Array_Resize(ar, ar->numElements);
}
