/*
=============================================================================
Module Information
------------------
Name:			cnt_strstream.cpp
Author:			Rich Whitehouse
Description:	container - stream
=============================================================================
*/

#include "cnt_arraylist.h"
#include "pluginshare.h"

typedef struct cntStream_s
{
	unsigned char		*buffer;
	bool				bufferOwned;
	int					bufPtr;
	int					bufPtrBits;
	int					size;
	int					padSize;
	int					growSize;

	void				*(*mem_malloc)(size_t size);
	void				*(*mem_realloc)(void *ptr, size_t newSize);
	void				(*mem_free)(void *ptr);
} cntStream_t;

//alloc
cntStream_t *Stream_Alloc(void *buffer, int size)
{
	cntStream_t *st = (cntStream_t *)malloc(sizeof(cntStream_t));
	memset(st, 0, sizeof(cntStream_t));

	st->mem_malloc = malloc;
	st->mem_realloc = realloc;
	st->mem_free = free;

	if (!buffer)
	{
		st->padSize = size;
		st->size = 0;
		st->buffer = (unsigned char *)st->mem_malloc(st->padSize);
		memset(st->buffer, 0, st->padSize);
		st->bufferOwned = true;
	}
	else
	{
		st->padSize = size;
		st->size = size;
		st->buffer = (unsigned char *)buffer;
	}
	st->growSize = 0;//1024;
	return st;
}

//free
void Stream_Free(cntStream_t *st)
{
	if (st->buffer && st->bufferOwned)
	{
		st->mem_free(st->buffer);
	}
	st->mem_free(st);
}

//check growth
static void Stream_CheckGrowth(cntStream_t *st, int maxOfs)
{
	if (maxOfs >= st->padSize)
	{
		assert(st->bufferOwned); //running out of space in a provided buffer would be bad
		int pad = st->padSize;
		while (maxOfs >= pad)
		{
			if (st->growSize)
			{
				pad += st->growSize;
			}
			else
			{ //default to exponential growth
				pad *= 2;
			}
		}
		unsigned char *buf = (unsigned char *)st->mem_malloc(pad);
		memcpy(buf, st->buffer, st->padSize);
		memset(buf+st->padSize, 0, pad-st->padSize);
		st->mem_free(st->buffer);
		st->buffer = buf;
		st->padSize = pad;
	}
	if (maxOfs > st->size)
	{
		st->size = maxOfs;
	}
}

//write byte
static void Stream_WriteByte(cntStream_t *st, unsigned char b)
{
	int bitByte = (st->bufPtrBits > 0) ? 1 : 0;
	Stream_CheckGrowth(st, st->bufPtr+bitByte);

	unsigned char *dst = st->buffer+st->bufPtr;
	if (!st->bufPtrBits)
	{
		*dst = b;
		st->bufPtr++;
	}
	else
	{
		*dst |= (b & ((1<<(8-st->bufPtrBits))-1))<<st->bufPtrBits;
		dst++;
		st->bufPtrBits = 8-st->bufPtrBits;
		b >>= st->bufPtrBits;
		*dst |= b;
		st->bufPtr += 2;
	}
}

//write bits
void Stream_WriteBits(cntStream_t *st, void *buf, int size)
{
	unsigned char *src = (unsigned char *)buf;
	while (size >= 8)
	{
		Stream_WriteByte(st, *src);
		src++;
		size -= 8;
	}
	if (size > 0)
	{ //write extra bits
		int bitByte = ((st->bufPtrBits+size) > 8) ? 1 : 0;
		Stream_CheckGrowth(st, st->bufPtr+bitByte);
		unsigned char *dst = st->buffer+st->bufPtr;
		unsigned char sval = ((*src) & ((1<<size)-1));
		if (!st->bufPtrBits)
		{
			*dst = sval;
			st->bufPtrBits = size;
		}
		else
		{
			*dst |= (sval & ((1<<(8-st->bufPtrBits))-1))<<st->bufPtrBits;
			//sval >>= (size-(8-st->bufPtrBits));
			sval >>= (8-st->bufPtrBits);
			st->bufPtrBits += size;
			if (st->bufPtrBits >= 8)
			{
				st->bufPtr++;
				st->bufPtrBits -= 8;
				if (st->bufPtrBits > 0)
				{
					dst++;
					*dst |= sval;
				}
			}
		}
	}
}

//write bytes
void Stream_WriteBytes(cntStream_t *st, void *buf, int size)
{
	if (!st->bufPtrBits)
	{
		Stream_CheckGrowth(st, st->bufPtr+size);
		memcpy(st->buffer+st->bufPtr, buf, size);
		st->bufPtr += size;
	}
	else
	{
		unsigned char *src = (unsigned char *)buf;
		for (int i = 0; i < size; i++)
		{
			Stream_WriteByte(st, src[i]);
		}
	}
}

//read byte
static bool Stream_ReadByte(cntStream_t *st, unsigned char *b)
{
	int bitByte = (st->bufPtrBits > 0) ? 1 : 0;
	if ((st->bufPtr+bitByte) >= st->size)
	{
		return false;
	}

	unsigned char *src = st->buffer+st->bufPtr;
	if (!st->bufPtrBits)
	{
		*b = *src;
		st->bufPtr++;
	}
	else
	{
		*b = ((*src)>>st->bufPtrBits);
		src++;
		*b |= ((*src) & ((1<<st->bufPtrBits)-1))<<(8-st->bufPtrBits);
		st->bufPtrBits = 8-st->bufPtrBits;
		st->bufPtr += 2;
	}
	return true;
}

//read bits
bool Stream_ReadBits(cntStream_t *st, void *buf, int size)
{
	unsigned char *dst = (unsigned char *)buf;
	while (size >= 8)
	{
		if (!Stream_ReadByte(st, dst))
		{
			return false;
		}
		dst++;
		size -= 8;
	}
	if (size > 0)
	{ //read extra bits
		if (st->bufPtr >= st->size)
		{
			return false;
		}
		unsigned char *src = st->buffer+st->bufPtr;
		if (!st->bufPtrBits)
		{
			*dst = ((*src) & ((1<<size)-1));
			st->bufPtrBits += size;
		}
		else
		{
			unsigned char sval = (*src)>>st->bufPtrBits;
			*dst = (sval & ((1<<size)-1));
			unsigned int ls = 8-st->bufPtrBits;
			st->bufPtrBits += size;
			if (st->bufPtrBits >= 8)
			{
				src++;
				st->bufPtr++;
				st->bufPtrBits -= 8;
				if (st->bufPtrBits > 0)
				{
					if (st->bufPtr >= st->size)
					{
						return false;
					}
					*dst |= ((*src) & ((1<<st->bufPtrBits)-1))<<ls;
				}
			}
		}
	}

	return true;
}

//write bytes
bool Stream_ReadBytes(cntStream_t *st, void *buf, int size)
{
	if (!st->bufPtrBits)
	{
		if ((st->bufPtr+size) >= st->size)
		{
			return false;
		}
		memcpy(buf, st->buffer+st->bufPtr, size);
		st->bufPtr += size;
	}
	else
	{
		unsigned char *dst = (unsigned char *)buf;
		for (int i = 0; i < size; i++)
		{
			if (!Stream_ReadByte(st, dst+i))
			{
				return false;
			}
		}
	}
	return true;
}

//write bool
void Stream_WriteBool(cntStream_t *st, bool val)
{
	Stream_WriteBytes(st, &val, sizeof(val));
}
//write int
void Stream_WriteInt(cntStream_t *st, int val)
{
	Stream_WriteBytes(st, &val, sizeof(val));
}
//write float
void Stream_WriteFloat(cntStream_t *st, float val)
{
	Stream_WriteBytes(st, &val, sizeof(val));
}
//write string
void Stream_WriteString(cntStream_t *st, char *str, bool noTerminate)
{
	int l = (int)strlen(str);
	if (!noTerminate)
	{
		l += 1;
	}
	Stream_WriteBytes(st, str, l);
}

//read bool
bool Stream_ReadBool(cntStream_t *st)
{
	bool val = false;
	Stream_ReadBytes(st, &val, sizeof(val));
	return val;
}
//read int
int Stream_ReadInt(cntStream_t *st)
{
	int val = 0;
	Stream_ReadBytes(st, &val, sizeof(val));
	return val;
}
//read float
float Stream_ReadFloat(cntStream_t *st)
{
	float val = 0.0f;
	Stream_ReadBytes(st, &val, sizeof(val));
	return val;
}
//read string
void Stream_ReadString(cntStream_t *st, char *str, int maxSize)
{
	int i;
	for (i = 0; i < maxSize-1; i++)
	{
		unsigned char b;
		if (!Stream_ReadByte(st, &b) || !b)
		{
			break;
		}
		str[i] = (char)b;
	}
	str[i] = 0;
}

//get buffer
void *Stream_Buffer(cntStream_t *st)
{
	return st->buffer;
}

//get size
int Stream_Size(cntStream_t *st)
{
	if (st->bufPtr > st->size)
	{
		return st->bufPtr;
	}
	return st->size;
}

//set offset
void Stream_SetOffset(cntStream_t *st, int offset)
{
	if (offset >= 0 && offset < st->size)
	{
		st->bufPtr = offset;
	}
}

//get offset
int Stream_GetOffset(cntStream_t *st)
{
	return st->bufPtr;
}
