#include "rm_common.h"

//===========================================
//RichAngles implementation
//===========================================
RichAngles::RichAngles(void)
{
	a[0] = 0.0f;
	a[1] = 0.0f;
	a[2] = 0.0f;
}
RichAngles::RichAngles(const float pitch, const float yaw, const float roll)
{
	a[0] = pitch;
	a[1] = yaw;
	a[2] = roll;
}
RichAngles::RichAngles(const float *angles)
{
	a[0] = angles[0];
	a[1] = angles[1];
	a[2] = angles[2];
}
RichAngles::RichAngles(const float *radians, bool scaleToDegrees)
{
	a[0] = radians[0];
	a[1] = radians[1];
	a[2] = radians[2];
	if (scaleToDegrees)
	{
		a[0] *= g_flRadToDeg;
		a[1] *= g_flRadToDeg;
		a[2] *= g_flRadToDeg;
	}
}
float &RichAngles::operator[](int idx)
{
	assert(idx >= 0 && idx < 3);
	return a[idx];
}
float RichAngles::operator[](int idx) const
{
	assert(idx >= 0 && idx < 3);
	return a[idx];
}
RichAngles &RichAngles::operator=(const RichAngles &angles)
{
	a[0] = angles.a[0];
	a[1] = angles.a[1];
	a[2] = angles.a[2];
	return *this;
}
bool RichAngles::operator==(const RichAngles &angles) const
{
	return (a[0] == angles.a[0] && a[1] == angles.a[1] && a[2] == angles.a[2]);
}
bool RichAngles::operator!=(const RichAngles &angles) const
{
	return (a[0] != angles.a[0] || a[1] != angles.a[1] || a[2] != angles.a[2]);
}
RichAngles RichAngles::operator+(const RichAngles &angles) const
{
	return RichAngles(a[0]+angles.a[0], a[1]+angles.a[1], a[2]+angles.a[2]);
}
RichAngles &RichAngles::operator+=(const RichAngles &angles)
{
	a[0] += angles.a[0];
	a[1] += angles.a[1];
	a[2] += angles.a[2];
	return *this;
}
RichAngles RichAngles::operator-(void) const
{
	return RichAngles(-a[0], -a[1], -a[2]);
}
RichAngles RichAngles::operator-(const RichAngles &angles) const
{
	return RichAngles(a[0]-angles.a[0], a[1]-angles.a[1], a[2]-angles.a[2]);
}
RichAngles &RichAngles::operator-=(const RichAngles &angles)
{
	a[0] -= angles.a[0];
	a[1] -= angles.a[1];
	a[2] -= angles.a[2];
	return *this;
}
RichAngles RichAngles::operator*(const RichAngles &angles) const
{
	return RichAngles(a[0]*angles.a[0], a[1]*angles.a[1], a[2]*angles.a[2]);
}
RichAngles &RichAngles::operator*=(const RichAngles &angles)
{
	a[0] *= angles.a[0];
	a[1] *= angles.a[1];
	a[2] *= angles.a[2];
	return *this;
}
RichAngles RichAngles::operator*(const float &f) const
{
	return RichAngles(a[0]*f, a[1]*f, a[2]*f);
}
RichAngles &RichAngles::operator*=(const float &f)
{
	a[0] *= f;
	a[1] *= f;
	a[2] *= f;
	return *this;
}
RichAngles RichAngles::operator/(const RichAngles &angles) const
{
	return RichAngles(a[0]/angles.a[0], a[1]/angles.a[1], a[2]/angles.a[2]);
}
RichAngles &RichAngles::operator/=(const RichAngles &angles)
{
	a[0] /= angles.a[0];
	a[1] /= angles.a[1];
	a[2] /= angles.a[2];
	return *this;
}

void RichAngles::ChangeEndian(void)
{
	LITTLE_BIG_SWAP(a[0]);
	LITTLE_BIG_SWAP(a[1]);
	LITTLE_BIG_SWAP(a[2]);
}

void RichAngles::Mod(float f)
{
	a[0] = fmodf(a[0], f);
	a[1] = fmodf(a[1], f);
	a[2] = fmodf(a[2], f);
}
void RichAngles::Normalize360(void)
{
	Mod(360.0f);
	for (int i = 0; i < 3; i++)
	{
		if (a[i] < 0.0f)
		{
			a[i] += 360.0f;
		}
	}
}
void RichAngles::Normalize180(void)
{
	Normalize360();
	for (int i = 0; i < 3; i++)
	{
		if (a[i] > 180.0f)
		{
			a[i] -= 360.0f;
		}
	}
}

void RichAngles::AngleVectors(RichVec3 *fwd, RichVec3 *right, RichVec3 *up) const
{
	float *v1 = (fwd) ? fwd->v : NULL;
	float *v2 = (right) ? right->v : NULL;
	float *v3 = (up) ? up->v : NULL;
	g_mfn->Math_AngleVectors((float *)a, v1, v2, v3);
}
void RichAngles::Lerp(const RichAngles &angles, const float frac)
{
	a[0] = g_mfn->Math_LinearLerp(a[0], angles.a[0], frac);
	a[1] = g_mfn->Math_LinearLerp(a[1], angles.a[1], frac);
	a[2] = g_mfn->Math_LinearLerp(a[2], angles.a[2], frac);
}
void RichAngles::Lerp(const RichAngles &anglesA, const RichAngles &anglesB, const float frac)
{
	a[0] = g_mfn->Math_LinearLerp(anglesA.a[0], anglesB.a[0], frac);
	a[1] = g_mfn->Math_LinearLerp(anglesA.a[1], anglesB.a[1], frac);
	a[2] = g_mfn->Math_LinearLerp(anglesA.a[2], anglesB.a[2], frac);
}
void RichAngles::ALerp(const RichAngles &angles, const float degrees)
{
	a[0] = g_mfn->Math_BlendAngleLinear(a[0], angles.a[0], degrees);
	a[1] = g_mfn->Math_BlendAngleLinear(a[1], angles.a[1], degrees);
	a[2] = g_mfn->Math_BlendAngleLinear(a[2], angles.a[2], degrees);
}
void RichAngles::ALerp(const RichAngles &anglesA, const RichAngles &anglesB, const float degrees)
{
	a[0] = g_mfn->Math_BlendAngleLinear(anglesA.a[0], anglesB.a[0], degrees);
	a[1] = g_mfn->Math_BlendAngleLinear(anglesA.a[1], anglesB.a[1], degrees);
	a[2] = g_mfn->Math_BlendAngleLinear(anglesA.a[2], anglesB.a[2], degrees);
}

RichVec3 RichAngles::ToVec3(void) const
{
	float t[3];
	g_mfn->Math_AngleVectors((float *)a, t, NULL, NULL);
	return RichVec3(t);
}
RichMat43 RichAngles::ToMat43(void) const
{
	modelMatrix_t m;
	g_mfn->Math_AnglesToMat(a, &m);
	return RichMat43(m);
}
RichMat43 RichAngles::ToMat43_XYZ(bool yFlip) const
{
	RichMat43 m;
	m.Rotate(a[2], 0.0f, 0.0f, 1.0f);
	float y = (yFlip) ? -a[1] : a[1];
	m.Rotate(y, 0.0f, 1.0f, 0.0f);
	m.Rotate(a[0], 1.0f, 0.0f, 0.0f);
	return m;
}
RichQuat RichAngles::ToQuat(void) const
{
	return ToMat43().ToQuat();
}
