#include "rm_common.h"

//===========================================
//RichMat43 implementation
//===========================================
RichMat43::RichMat43(void)
{
	m = g_identityMatrix;
}
RichMat43::RichMat43(const RichVec3 &r0, const RichVec3 &r1, const RichVec3 &r2, const RichVec3 &trans)
{
	RichVec3 *mv = (RichVec3 *)&m;
	mv[0] = r0;
	mv[1] = r1;
	mv[2] = r2;
	mv[3] = trans;
}
RichMat43::RichMat43(const modelMatrix_t &mat)
{
	m = mat;
}
RichMat43::RichMat43(const float *mat)
{
	const modelMatrix_t *t = (const modelMatrix_t *)mat;
	m = *t;
}

RichVec3 &RichMat43::operator[](int idx)
{
	assert(idx >= 0 && idx < 4);
	RichVec3 *mv = (RichVec3 *)&m;
	return mv[idx];
}
const RichVec3 &RichMat43::operator[](int idx) const
{
	assert(idx >= 0 && idx < 4);
	const RichVec3 *mv = (const RichVec3 *)&m;
	return mv[idx];
}

RichMat43 &RichMat43::operator=(const RichMat43 &mat)
{
	m = mat.m;
	return *this;
}
RichMat43 &RichMat43::operator=(const modelMatrix_t &mat)
{
	m = mat;
	return *this;
}
bool RichMat43::operator==(const RichMat43 &mat) const
{
	const RichVec3 *mvA = (const RichVec3 *)&m;
	const RichVec3 *mvB = (const RichVec3 *)&mat.m;
	return (mvA[0] == mvB[0] && mvA[1] == mvB[1] && mvA[2] == mvB[2] && mvA[3] == mvB[3]);
}
bool RichMat43::operator!=(const RichMat43 &mat) const
{
	const RichVec3 *mvA = (const RichVec3 *)&m;
	const RichVec3 *mvB = (const RichVec3 *)&mat.m;
	return (mvA[0] != mvB[0] || mvA[1] != mvB[1] || mvA[2] != mvB[2] || mvA[3] != mvB[3]);
}
RichMat43 RichMat43::operator+(const RichMat43 &mat) const
{
	const RichVec3 *mvA = (const RichVec3 *)&m;
	const RichVec3 *mvB = (const RichVec3 *)&mat.m;

	return RichMat43(mvA[0]+mvB[0], mvA[1]+mvB[1], mvA[2]+mvB[2], mvA[3]+mvB[3]);
}
RichMat43 &RichMat43::operator+=(const RichMat43 &mat)
{
	RichVec3 *mvA = (RichVec3 *)&m;
	const RichVec3 *mvB = (const RichVec3 *)&mat.m;

	mvA[0] += mvB[0];
	mvA[1] += mvB[1];
	mvA[2] += mvB[2];
	mvA[3] += mvB[3];
	return *this;
}
RichMat43 RichMat43::operator-(void) const
{
	const RichVec3 *mv = (const RichVec3 *)&m;

	return RichMat43(-mv[0], -mv[1], -mv[2], -mv[3]);
}
RichMat43 RichMat43::operator-(const RichMat43 &mat) const
{
	const RichVec3 *mvA = (const RichVec3 *)&m;
	const RichVec3 *mvB = (const RichVec3 *)&mat.m;

	return RichMat43(mvA[0]-mvB[0], mvA[1]-mvB[1], mvA[2]-mvB[2], mvA[3]-mvB[3]);
}
RichMat43 &RichMat43::operator-=(const RichMat43 &mat)
{
	RichVec3 *mvA = (RichVec3 *)&m;
	const RichVec3 *mvB = (const RichVec3 *)&mat.m;

	mvA[0] -= mvB[0];
	mvA[1] -= mvB[1];
	mvA[2] -= mvB[2];
	mvA[3] -= mvB[3];
	return *this;
}
RichMat43 RichMat43::operator*(const RichMat43 &mat) const
{
	modelMatrix_t t;
	g_mfn->Math_MatrixMultiply((modelMatrix_t *)&mat.m, (modelMatrix_t *)&m, &t);
	return RichMat43(t);
}
RichMat43 &RichMat43::operator*=(const RichMat43 &mat)
{
	modelMatrix_t t = m;
	g_mfn->Math_MatrixMultiply((modelMatrix_t *)&mat.m, &t, &m);
	return *this;
}
RichVec3 RichMat43::operator*(const RichVec3 &vec) const
{
	return TransformPoint(vec);
}
RichVec4 RichMat43::operator*(const RichVec4 &vec) const
{
	return TransformVec4(vec);
}

void RichMat43::ChangeEndian(void)
{
	RichVec3 *mv = (RichVec3 *)&m;
	mv[0].ChangeEndian();
	mv[1].ChangeEndian();
	mv[2].ChangeEndian();
	mv[3].ChangeEndian();
}

RichVec3 RichMat43::TransformPoint(const RichVec3 &vec) const
{
	float t[3];
	g_mfn->Math_TransformPointByMatrix((modelMatrix_t *)&m, (float *)vec.v, t);
	return RichVec3(t);
}
RichVec3 RichMat43::TransformNormal(const RichVec3 &vec) const
{
	float t[3];
	g_mfn->Math_TransformPointByMatrixNoTrans((modelMatrix_t *)&m, (float *)vec.v, t);
	return RichVec3(t);
}
RichVec4 RichMat43::TransformVec4(const RichVec4 &vec) const
{
	return ToMat44().TransformVec4(vec);
}
RichMat43 RichMat43::GetTranspose(void) const
{
	modelMatrix_t t;
	g_mfn->Math_TransposeMat((modelMatrix_t *)&m, &t);
	return RichMat43(t);
}
void RichMat43::Transpose(void)
{
	modelMatrix_t t;
	g_mfn->Math_TransposeMat(&m, &t);
	m = t;
}
RichMat43 RichMat43::GetInverse(void) const
{
	modelMatrix_t t;
	g_mfn->Math_MatrixInverse((modelMatrix_t *)&m, &t);
	return RichMat43(t);
}
void RichMat43::Inverse(void)
{
	modelMatrix_t t;
	g_mfn->Math_MatrixInverse(&m, &t);
	m = t;
}
RichMat43 RichMat43::GetOrthogonalize(bool keepScale, bool keepFlip, bool straightCross) const
{
	modelMatrix_t t = m;
	g_mfn->Math_OrthogonalizeMatrix(&t, keepScale, keepFlip, straightCross);
	return RichMat43(t);
}
void RichMat43::Orthogonalize(bool keepScale, bool keepFlip, bool straightCross)
{
	g_mfn->Math_OrthogonalizeMatrix(&m, keepScale, keepFlip, straightCross);
}
bool RichMat43::IsSkewed(void)
{
	return g_mfn->Math_MatrixIsSkewed(&m);
}
void RichMat43::Rotate(float degrees, float x, float y, float z, bool transposeRot)
{
	if (transposeRot)
	{
		g_mfn->Math_RotateMatrix(&m, degrees, x, y, z);
	}
	else
	{
		g_mfn->Math_RotateMatrixTP(&m, degrees, x, y, z);
	}
}
void RichMat43::Rotate(float degrees, float *xyz, bool transposeRot)
{
	if (transposeRot)
	{
		g_mfn->Math_RotateMatrix(&m, degrees, xyz[0], xyz[1], xyz[2]);
	}
	else
	{
		g_mfn->Math_RotateMatrixTP(&m, degrees, xyz[0], xyz[1], xyz[2]);
	}
}
void RichMat43::Translate(float x, float y, float z)
{
	float xyz[3] = {x, y, z};
	g_mfn->Math_TranslateMatrix(&m, xyz);
}
void RichMat43::Translate(float *xyz)
{
	g_mfn->Math_TranslateMatrix(&m, xyz);
}
void RichMat43::Lerp(const RichMat43 &postMat, float lerpFrac, bool nonUniform, bool orthogonalize)
{
	modelMatrix_t mout;
	g_mfn->Math_LerpMatrices((modelMatrix_t &)m, (modelMatrix_t &)postMat.m, 1.0f-lerpFrac, mout, nonUniform, orthogonalize);
	m = mout;
}
void RichMat43::Lerp(const RichMat43 &preMat, const RichMat43 &postMat, float lerpFrac, bool nonUniform, bool orthogonalize)
{
	g_mfn->Math_LerpMatrices((modelMatrix_t &)preMat.m, (modelMatrix_t &)postMat.m, 1.0f-lerpFrac, m, nonUniform, orthogonalize);
}
void RichMat43::SLerp(const RichMat43 &postMat, float lerpFrac, bool nonUniform)
{
	modelMatrix_t mout;
	g_mfn->Math_LerpMatricesQ((modelMatrix_t &)m, (modelMatrix_t &)postMat.m, 1.0f-lerpFrac, mout, nonUniform);
	m = mout;
}
void RichMat43::SLerp(const RichMat43 &preMat, const RichMat43 &postMat, float lerpFrac, bool nonUniform)
{
	g_mfn->Math_LerpMatricesQ((modelMatrix_t &)preMat.m, (modelMatrix_t &)postMat.m, 1.0f-lerpFrac, m, nonUniform);
}

RichQuat RichMat43::ToQuat(void) const
{
	float t[4];
	g_mfn->Math_MatToQuat((modelMatrix_t *)&m, t, false);
	return RichQuat(t);
}
RichAngles RichMat43::ToAngles(void) const
{
	float t[3];
	g_mfn->Math_MatToAngles(t, &m);
	RichAngles ang(t);
	ang.Normalize360();
	return ang;
}
RichMat44 RichMat43::ToMat44(void) const
{
	fourxMatrix_t mat;
	g_mfn->Math_ModelMatToGL((modelMatrix_t *)&m, (float *)&mat);
	return RichMat44(mat);
}


//===========================================
//RichMat44 implementation
//===========================================
RichMat44::RichMat44(void)
{
	m = g_identityMatrix4x4;
}
RichMat44::RichMat44(const RichVec4 &r0, const RichVec4 &r1, const RichVec4 &r2, const RichVec4 &r3)
{
	RichVec4 *mv = (RichVec4 *)&m;
	mv[0] = r0;
	mv[1] = r1;
	mv[2] = r2;
	mv[2] = r3;
}
RichMat44::RichMat44(const fourxMatrix_t &mat)
{
	m = mat;
}
RichMat44::RichMat44(const float *mat)
{
	const fourxMatrix_t *t = (const fourxMatrix_t *)mat;
	m = *t;
}

RichVec4 &RichMat44::operator[](int idx)
{
	assert(idx >= 0 && idx < 4);
	RichVec4 *mv = (RichVec4 *)&m;
	return mv[idx];
}
const RichVec4 &RichMat44::operator[](int idx) const
{
	assert(idx >= 0 && idx < 4);
	const RichVec4 *mv = (const RichVec4 *)&m;
	return mv[idx];
}

RichMat44 &RichMat44::operator=(const RichMat44 &mat)
{
	m = mat.m;
	return *this;
}
RichMat44 &RichMat44::operator=(const fourxMatrix_t &mat)
{
	m = mat;
	return *this;
}
bool RichMat44::operator==(const RichMat44 &mat) const
{
	const RichVec4 *mvA = (const RichVec4 *)&m;
	const RichVec4 *mvB = (const RichVec4 *)&mat.m;
	return (mvA[0] == mvB[0] && mvA[1] == mvB[1] && mvA[2] == mvB[2] && mvA[3] == mvB[3]);
}
bool RichMat44::operator!=(const RichMat44 &mat) const
{
	const RichVec4 *mvA = (const RichVec4 *)&m;
	const RichVec4 *mvB = (const RichVec4 *)&mat.m;
	return (mvA[0] != mvB[0] || mvA[1] != mvB[1] || mvA[2] != mvB[2] || mvA[3] != mvB[3]);
}
RichMat44 RichMat44::operator+(const RichMat44 &mat) const
{
	const RichVec4 *mvA = (const RichVec4 *)&m;
	const RichVec4 *mvB = (const RichVec4 *)&mat.m;

	return RichMat44(mvA[0]+mvB[0], mvA[1]+mvB[1], mvA[2]+mvB[2], mvA[3]+mvB[3]);
}
RichMat44 &RichMat44::operator+=(const RichMat44 &mat)
{
	RichVec4 *mvA = (RichVec4 *)&m;
	const RichVec4 *mvB = (const RichVec4 *)&mat.m;

	mvA[0] += mvB[0];
	mvA[1] += mvB[1];
	mvA[2] += mvB[2];
	mvA[3] += mvB[3];
	return *this;
}
RichMat44 RichMat44::operator-(void) const
{
	const RichVec4 *mv = (const RichVec4 *)&m;

	return RichMat44(-mv[0], -mv[1], -mv[2], -mv[3]);
}
RichMat44 RichMat44::operator-(const RichMat44 &mat) const
{
	const RichVec4 *mvA = (const RichVec4 *)&m;
	const RichVec4 *mvB = (const RichVec4 *)&mat.m;

	return RichMat44(mvA[0]-mvB[0], mvA[1]-mvB[1], mvA[2]-mvB[2], mvA[3]-mvB[3]);
}
RichMat44 &RichMat44::operator-=(const RichMat44 &mat)
{
	RichVec4 *mvA = (RichVec4 *)&m;
	const RichVec4 *mvB = (const RichVec4 *)&mat.m;

	mvA[0] -= mvB[0];
	mvA[1] -= mvB[1];
	mvA[2] -= mvB[2];
	mvA[3] -= mvB[3];
	return *this;
}
RichMat44 RichMat44::operator*(const RichMat44 &mat) const
{
	fourxMatrix_t t;
	g_mfn->Math_MatrixMultiply4x4((fourxMatrix_t *)&m, (fourxMatrix_t *)&mat.m, &t);
	return RichMat44(t);
}
RichMat44 &RichMat44::operator*=(const RichMat44 &mat)
{
	fourxMatrix_t t = m;
	g_mfn->Math_MatrixMultiply4x4(&t, (fourxMatrix_t *)&mat.m, &m);
	return *this;
}
RichVec4 RichMat44::operator*(const RichVec4 &vec) const
{
	return TransformVec4(vec);
}

void RichMat44::ChangeEndian(void)
{
	RichVec4 *mv = (RichVec4 *)&m;
	mv[0].ChangeEndian();
	mv[1].ChangeEndian();
	mv[2].ChangeEndian();
	mv[3].ChangeEndian();
}

RichVec4 RichMat44::TransformVec4(const RichVec4 &vec) const
{
	float out[4];
	const float *matrix = (float *)&m;
	const float *in = vec.v;
	out[0] = matrix[0]*in[0] + matrix[4+0]*in[1] + matrix[8+0]*in[2] + matrix[12+0]*in[3];
	out[1] = matrix[1]*in[0] + matrix[4+1]*in[1] + matrix[8+1]*in[2] + matrix[12+1]*in[3];
	out[2] = matrix[2]*in[0] + matrix[4+2]*in[1] + matrix[8+2]*in[2] + matrix[12+2]*in[3];
	out[3] = matrix[3]*in[0] + matrix[4+3]*in[1] + matrix[8+3]*in[2] + matrix[12+3]*in[3];
	return RichVec4(out);
}
RichMat44 RichMat44::GetTranspose(void) const
{
	RichMat44 dst;
	const RichMat44 &src = *this;
	for (int i = 0; i < 4; i++)
	{
		for (int j = 0; j < 4; j++)
		{ //swap rows/columns
			dst[i][j] = src[j][i];
		}
	}
	return dst;
}
void RichMat44::Transpose(void)
{
	*this = GetTranspose();
}
RichMat44 RichMat44::GetInverse(void) const
{
	fourxMatrix_t t;
	g_mfn->Math_MatrixInverse4x4((float *)&m, (float *)&t);
	return RichMat44(t);
}
void RichMat44::Inverse(void)
{
	fourxMatrix_t t = m;
	g_mfn->Math_MatrixInverse4x4((float *)&t, (float *)&m);
}
void RichMat44::Rotate(float degrees, float x, float y, float z)
{
	g_mfn->Math_RotateMatrix4x4(&m, degrees, x, y, z);
}
void RichMat44::Rotate(float degrees, float *xyz)
{
	g_mfn->Math_RotateMatrix4x4(&m, degrees, xyz[0], xyz[1], xyz[2]);
}
void RichMat44::Translate(float x, float y, float z)
{
	float xyz[3] = {x, y, z};
	g_mfn->Math_TranslateMatrix4x4(&m, xyz);
}
void RichMat44::Translate(float *xyz)
{
	g_mfn->Math_TranslateMatrix4x4(&m, xyz);
}

RichMat43 RichMat44::ToMat43(void) const
{
	modelMatrix_t mat;
	g_mfn->Math_ModelMatFromGL(&mat, (float *)&m);
	return RichMat43(mat);
}
