#include "rm_common.h"

//===========================================
//RichQuat implementation
//===========================================
RichQuat::RichQuat(void)
{
	q[0] = 0.0f;
	q[1] = 0.0f;
	q[2] = 0.0f;
	q[3] = 0.0f;
}
RichQuat::RichQuat(const float x, const float y, const float z, const float w)
{
	q[0] = x;
	q[1] = y;
	q[2] = z;
	q[3] = w;
}
RichQuat::RichQuat(const float *xyzw)
{
	q[0] = xyzw[0];
	q[1] = xyzw[1];
	q[2] = xyzw[2];
	q[3] = xyzw[3];
}
RichQuat::RichQuat(const float *xyz, const bool noW)
{
	if (noW)
	{
		FromQuat3(xyz);
	}
	else
	{
		q[0] = xyz[0];
		q[1] = xyz[1];
		q[2] = xyz[2];
		q[3] = xyz[3];
	}
}

float &RichQuat::operator[](int idx)
{
	assert(idx >= 0 && idx < 4);
	return q[idx];
}
float RichQuat::operator[](int idx) const
{
	assert(idx >= 0 && idx < 4);
	return q[idx];
}
RichQuat &RichQuat::operator=(const RichQuat &quat)
{
	q[0] = quat.q[0];
	q[1] = quat.q[1];
	q[2] = quat.q[2];
	q[3] = quat.q[3];
	return *this;
}
bool RichQuat::operator==(const RichQuat &quat) const
{
	return (q[0] == quat.q[0] && q[1] == quat.q[1] && q[2] == quat.q[2] && q[3] == quat.q[3]);
}
bool RichQuat::operator!=(const RichQuat &quat) const
{
	return (q[0] != quat.q[0] || q[1] != quat.q[1] || q[2] != quat.q[2] || q[3] != quat.q[3]);
}
RichQuat RichQuat::operator+(const RichQuat &quat) const
{
	return RichQuat(q[0]+quat.q[0], q[1]+quat.q[1], q[2]+quat.q[2], q[3]+quat.q[3]);
}
RichQuat &RichQuat::operator+=(const RichQuat &quat)
{
	q[0] += quat.q[0];
	q[1] += quat.q[1];
	q[2] += quat.q[2];
	q[3] += quat.q[3];
	return *this;
}
RichQuat RichQuat::operator-(void) const
{
	return RichQuat(-q[0], -q[1], -q[2], -q[3]);
}
RichQuat RichQuat::operator-(const RichQuat &quat) const
{
	return RichQuat(q[0]-quat.q[0], q[1]-quat.q[1], q[2]-quat.q[2], q[3]-quat.q[3]);
}
RichQuat &RichQuat::operator-=(const RichQuat &quat)
{
	q[0] -= quat.q[0];
	q[1] -= quat.q[1];
	q[2] -= quat.q[2];
	q[3] -= quat.q[3];
	return *this;
}
RichQuat RichQuat::operator*(const RichQuat &quat) const
{
	float t[4];
	t[0] = q[3]*quat.q[0] + q[0]*quat.q[3] + q[1]*quat.q[2] - q[2]*quat.q[1];
	t[1] = q[3]*quat.q[1] + q[1]*quat.q[3] + q[2]*quat.q[0] - q[0]*quat.q[2];
	t[2] = q[3]*quat.q[2] + q[2]*quat.q[3] + q[0]*quat.q[1] - q[1]*quat.q[0];
	t[3] = q[3]*quat.q[3] - q[0]*quat.q[0] - q[1]*quat.q[1] - q[2]*quat.q[2];
	return RichQuat(t);
}
RichQuat &RichQuat::operator*=(const RichQuat &quat)
{
	float t[4];
	t[0] = q[3]*quat.q[0] + q[0]*quat.q[3] + q[1]*quat.q[2] - q[2]*quat.q[1];
	t[1] = q[3]*quat.q[1] + q[1]*quat.q[3] + q[2]*quat.q[0] - q[0]*quat.q[2];
	t[2] = q[3]*quat.q[2] + q[2]*quat.q[3] + q[0]*quat.q[1] - q[1]*quat.q[0];
	t[3] = q[3]*quat.q[3] - q[0]*quat.q[0] - q[1]*quat.q[1] - q[2]*quat.q[2];
	q[0] = t[0];
	q[1] = t[1];
	q[2] = t[2];
	q[3] = t[3];
	return *this;
}
RichVec3 RichQuat::operator*(const RichVec3 &vec) const
{
	return TransformPoint(vec);
}
RichQuat RichQuat::operator*(const float &f) const
{
	return RichQuat(q[0]*f, q[1]*f, q[2]*f, q[3]*f);
}
RichQuat &RichQuat::operator*=(const float &f)
{
	q[0] *= f;
	q[1] *= f;
	q[2] *= f;
	q[3] *= f;
	return *this;
}

void RichQuat::ChangeEndian(void)
{
	LITTLE_BIG_SWAP(q[0]);
	LITTLE_BIG_SWAP(q[1]);
	LITTLE_BIG_SWAP(q[2]);
	LITTLE_BIG_SWAP(q[3]);
}

RichVec3 RichQuat::TransformPoint(const RichVec3 &vec) const
{
	return ToMat43()*vec;
}
RichQuat RichQuat::GetTranspose(void)
{
	RichMat43 m = ToMat43(true);
	return m.ToQuat();
}
void RichQuat::Transpose(void)
{
	RichMat43 m = ToMat43(true);
	*this = m.ToQuat();
}
float RichQuat::Length(void) const
{
	return sqrtf(q[0]*q[0] + q[1]*q[1] + q[2]*q[2] + q[3]*q[3]);
}
float RichQuat::Normalize(void)
{
	float l = Length();
	if (l != 0.0f)
	{
		float lm = 1.0f/l;
		*this *= lm;
	}
	return l;
}
void RichQuat::Lerp(const RichQuat &quat, const float frac)
{
	q[0] = g_mfn->Math_LinearLerp(q[0], quat.q[0], frac);
	q[1] = g_mfn->Math_LinearLerp(q[1], quat.q[1], frac);
	q[2] = g_mfn->Math_LinearLerp(q[2], quat.q[2], frac);
	q[3] = g_mfn->Math_LinearLerp(q[3], quat.q[3], frac);
}
void RichQuat::Lerp(const RichQuat &quatA, const RichQuat &quatB, const float frac)
{
	q[0] = g_mfn->Math_LinearLerp(quatA.q[0], quatB.q[0], frac);
	q[1] = g_mfn->Math_LinearLerp(quatA.q[1], quatB.q[1], frac);
	q[2] = g_mfn->Math_LinearLerp(quatA.q[2], quatB.q[2], frac);
	q[3] = g_mfn->Math_LinearLerp(quatA.q[3], quatB.q[3], frac);
}
void RichQuat::SLerp(const RichQuat &quat, const float frac)
{
	float t[4];
	g_mfn->Math_QuatSlerp(q, (float *)quat.q, frac, t);
	q[0] = t[0];
	q[1] = t[1];
	q[2] = t[2];
	q[3] = t[3];
}
void RichQuat::SLerp(const RichQuat &quatA, const RichQuat &quatB, const float frac)
{
	g_mfn->Math_QuatSlerp((float *)quatA.q, (float *)quatB.q, frac, q);
}

void RichQuat::FromQuat3(const float *quat)
{
	q[0] = quat[0];
	q[1] = quat[1];
	q[2] = quat[2];
	q[3] = sqrtf(fabsf(1.0f - (quat[0]*quat[0] + quat[1]*quat[1] + quat[2]*quat[2])));
}
void RichQuat::ToQuat3(float *quat) const
{
	if (q[3] < 0.0f)
	{
		quat[0] = -q[0];
		quat[1] = -q[1];
		quat[2] = -q[2];
	}
	else
	{
		quat[0] = q[0];
		quat[1] = q[1];
		quat[2] = q[2];
	}
}
RichMat43 RichQuat::ToMat43(bool transposed) const
{
	modelMatrix_t m;
	g_mfn->Math_QuatToMat((float *)q, &m, false, transposed);
	return RichMat43(m);
}
RichAngles RichQuat::ToAngles(void) const
{
	return ToMat43().ToAngles();
}
