#include "rm_common.h"

//===========================================
//RichVec3 implementation
//===========================================
RichVec3::RichVec3(void)
{
	v[0] = 0.0f;
	v[1] = 0.0f;
	v[2] = 0.0f;
}
RichVec3::RichVec3(const float x, const float y, const float z)
{
	v[0] = x;
	v[1] = y;
	v[2] = z;
}
RichVec3::RichVec3(const float *xyz)
{
	v[0] = xyz[0];
	v[1] = xyz[1];
	v[2] = xyz[2];
}

float &RichVec3::operator[](int idx)
{
	assert(idx >= 0 && idx < 3);
	return v[idx];
}
float RichVec3::operator[](int idx) const
{
	assert(idx >= 0 && idx < 3);
	return v[idx];
}
RichVec3 &RichVec3::operator=(const RichVec3 &vec)
{
	v[0] = vec.v[0];
	v[1] = vec.v[1];
	v[2] = vec.v[2];
	return *this;
}
bool RichVec3::operator==(const RichVec3 &vec) const
{
	return (v[0] == vec.v[0] && v[1] == vec.v[1] && v[2] == vec.v[2]);
}
bool RichVec3::operator!=(const RichVec3 &vec) const
{
	return (v[0] != vec.v[0] || v[1] != vec.v[1] || v[2] != vec.v[2]);
}
RichVec3 RichVec3::operator+(const RichVec3 &vec) const
{
	return RichVec3(v[0]+vec.v[0], v[1]+vec.v[1], v[2]+vec.v[2]);
}
RichVec3 &RichVec3::operator+=(const RichVec3 &vec)
{
	v[0] += vec.v[0];
	v[1] += vec.v[1];
	v[2] += vec.v[2];
	return *this;
}
RichVec3 RichVec3::operator-(void) const
{
	return RichVec3(-v[0], -v[1], -v[2]);
}
RichVec3 RichVec3::operator-(const RichVec3 &vec) const
{
	return RichVec3(v[0]-vec.v[0], v[1]-vec.v[1], v[2]-vec.v[2]);
}
RichVec3 &RichVec3::operator-=(const RichVec3 &vec)
{
	v[0] -= vec.v[0];
	v[1] -= vec.v[1];
	v[2] -= vec.v[2];
	return *this;
}
RichVec3 RichVec3::operator*(const RichVec3 &vec) const
{
	return RichVec3(v[0]*vec.v[0], v[1]*vec.v[1], v[2]*vec.v[2]);
}
RichVec3 &RichVec3::operator*=(const RichVec3 &vec)
{
	v[0] *= vec.v[0];
	v[1] *= vec.v[1];
	v[2] *= vec.v[2];
	return *this;
}
RichVec3 RichVec3::operator*(const float &f) const
{
	return RichVec3(v[0]*f, v[1]*f, v[2]*f);
}
RichVec3 &RichVec3::operator*=(const float &f)
{
	v[0] *= f;
	v[1] *= f;
	v[2] *= f;
	return *this;
}
RichVec3 RichVec3::operator/(const RichVec3 &vec) const
{
	return RichVec3(v[0]/vec.v[0], v[1]/vec.v[1], v[2]/vec.v[2]);
}
RichVec3 &RichVec3::operator/=(const RichVec3 &vec)
{
	v[0] /= vec.v[0];
	v[1] /= vec.v[1];
	v[2] /= vec.v[2];
	return *this;
}

void RichVec3::ChangeEndian(void)
{
	LITTLE_BIG_SWAP(v[0]);
	LITTLE_BIG_SWAP(v[1]);
	LITTLE_BIG_SWAP(v[2]);
}

float RichVec3::Dot(const RichVec3 &vec) const
{
	return g_mfn->Math_DotProduct(v, vec.v);
}
RichVec3 RichVec3::Cross(const RichVec3 &vec) const
{
	float r[3];
	g_mfn->Math_CrossProduct(v, vec.v, r);
	return RichVec3(r);
}
void RichVec3::Cross(const RichVec3 &vecA, const RichVec3 &vecB)
{
	g_mfn->Math_CrossProduct(vecA.v, vecB.v, v);
}
float RichVec3::Length(void) const
{
	return sqrtf(LengthSq());
}
float RichVec3::LengthSq(void) const
{
	return (v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}
float RichVec3::Normalize(void)
{
	return g_mfn->Math_VecNorm(v);
}
RichVec3 RichVec3::Normalize(void) const
{
	float t[3];
	t[0] = v[0];
	t[1] = v[1];
	t[2] = v[2];
	g_mfn->Math_VecNorm(t);
	return RichVec3(t);
}
void RichVec3::Lerp(const RichVec3 &vec, const float frac)
{
	v[0] = g_mfn->Math_LinearLerp(v[0], vec.v[0], frac);
	v[1] = g_mfn->Math_LinearLerp(v[1], vec.v[1], frac);
	v[2] = g_mfn->Math_LinearLerp(v[2], vec.v[2], frac);
}
void RichVec3::Lerp(const RichVec3 &vecA, const RichVec3 &vecB, const float frac)
{
	v[0] = g_mfn->Math_LinearLerp(vecA.v[0], vecB.v[0], frac);
	v[1] = g_mfn->Math_LinearLerp(vecA.v[1], vecB.v[1], frac);
	v[2] = g_mfn->Math_LinearLerp(vecA.v[2], vecB.v[2], frac);
}

RichAngles RichVec3::ToAngles(void) const
{
	float t[3];
	g_mfn->Math_VecToAngles(v, t);
	return RichAngles(t);
}
RichVec4 RichVec3::ToVec4(void) const
{
	return RichVec4(v[0], v[1], v[2], 0.0f);
}
RichMat43 RichVec3::ToMat43(void) const
{
	RichVec3 trans;

	RichVec3 fwd = *this;
	RichVec3 right;
	float xy = v[0]*v[0] + v[1]*v[1];
	if (xy)
	{
		float ixyl = 1.0f/sqrtf(xy);
		right[0] = -v[1]*ixyl;
		right[1] = v[0]*ixyl;
	}
	else
	{
		right[0] = 1.0f;
	}
	RichVec3 up = fwd.Cross(right);
	return RichMat43(fwd, right, up, trans);
}


//===========================================
//RichVec4 implementation
//===========================================
RichVec4::RichVec4(void)
{
	v[0] = 0.0f;
	v[1] = 0.0f;
	v[2] = 0.0f;
	v[3] = 0.0f;
}
RichVec4::RichVec4(const float x, const float y, const float z, const float w)
{
	v[0] = x;
	v[1] = y;
	v[2] = z;
	v[3] = w;
}
RichVec4::RichVec4(const float *xyzw)
{
	v[0] = xyzw[0];
	v[1] = xyzw[1];
	v[2] = xyzw[2];
	v[3] = xyzw[3];
}
RichVec4::RichVec4(const RichVec3 &xyz, const float w)
{
	v[0] = xyz.v[0];
	v[1] = xyz.v[1];
	v[2] = xyz.v[2];
	v[3] = w;
}

float &RichVec4::operator[](int idx)
{
	assert(idx >= 0 && idx < 4);
	return v[idx];
}
float RichVec4::operator[](int idx) const
{
	assert(idx >= 0 && idx < 4);
	return v[idx];
}
RichVec4 &RichVec4::operator=(const RichVec4 &vec)
{
	v[0] = vec.v[0];
	v[1] = vec.v[1];
	v[2] = vec.v[2];
	v[3] = vec.v[3];
	return *this;
}
bool RichVec4::operator==(const RichVec4 &vec) const
{
	return (v[0] == vec.v[0] && v[1] == vec.v[1] && v[2] == vec.v[2] && v[3] == vec.v[3]);
}
bool RichVec4::operator!=(const RichVec4 &vec) const
{
	return (v[0] != vec.v[0] || v[1] != vec.v[1] || v[2] != vec.v[2] || v[3] != vec.v[3]);
}
RichVec4 RichVec4::operator+(const RichVec4 &vec) const
{
	return RichVec4(v[0]+vec.v[0], v[1]+vec.v[1], v[2]+vec.v[2], v[3]+vec.v[3]);
}
RichVec4 &RichVec4::operator+=(const RichVec4 &vec)
{
	v[0] += vec.v[0];
	v[1] += vec.v[1];
	v[2] += vec.v[2];
	v[3] += vec.v[3];
	return *this;
}
RichVec4 RichVec4::operator-(void) const
{
	return RichVec4(-v[0], -v[1], -v[2], -v[3]);
}
RichVec4 RichVec4::operator-(const RichVec4 &vec) const
{
	return RichVec4(v[0]-vec.v[0], v[1]-vec.v[1], v[2]-vec.v[2], v[3]-vec.v[3]);
}
RichVec4 &RichVec4::operator-=(const RichVec4 &vec)
{
	v[0] -= vec.v[0];
	v[1] -= vec.v[1];
	v[2] -= vec.v[2];
	v[3] -= vec.v[3];
	return *this;
}
RichVec4 RichVec4::operator*(const RichVec4 &vec) const
{
	return RichVec4(v[0]*vec.v[0], v[1]*vec.v[1], v[2]*vec.v[2], v[3]*vec.v[3]);
}
RichVec4 &RichVec4::operator*=(const RichVec4 &vec)
{
	v[0] *= vec.v[0];
	v[1] *= vec.v[1];
	v[2] *= vec.v[2];
	v[3] *= vec.v[3];
	return *this;
}
RichVec4 RichVec4::operator*(const float &f) const
{
	return RichVec4(v[0]*f, v[1]*f, v[2]*f, v[3]*f);
}
RichVec4 &RichVec4::operator*=(const float &f)
{
	v[0] *= f;
	v[1] *= f;
	v[2] *= f;
	v[3] *= f;
	return *this;
}
RichVec4 RichVec4::operator/(const RichVec4 &vec) const
{
	return RichVec4(v[0]/vec.v[0], v[1]/vec.v[1], v[2]/vec.v[2], v[3]/vec.v[2]);
}
RichVec4 &RichVec4::operator/=(const RichVec4 &vec)
{
	v[0] /= vec.v[0];
	v[1] /= vec.v[1];
	v[2] /= vec.v[2];
	v[3] /= vec.v[3];
	return *this;
}

void RichVec4::ChangeEndian(void)
{
	LITTLE_BIG_SWAP(v[0]);
	LITTLE_BIG_SWAP(v[1]);
	LITTLE_BIG_SWAP(v[2]);
	LITTLE_BIG_SWAP(v[3]);
}

float RichVec4::Dot(const RichVec4 &vec) const
{
	const float *v1 = v;
	const float *v2 = vec.v;
	return v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2] + v1[3]*v2[3];
}
float RichVec4::Length(void) const
{
	return sqrtf(LengthSq());
}
float RichVec4::LengthSq(void) const
{
	return (v[0]*v[0] + v[1]*v[1] + v[2]*v[2] + v[3]*v[3]);
}
float RichVec4::Normalize(void)
{
	float l = Length();
	if (l != 0.0f)
	{
		float lm = 1.0f/l;
		*this *= lm;
	}
	return l;
}
RichVec4 RichVec4::Normalize(void) const
{
	float l = Length();
	if (l != 0.0f)
	{
		float lm = 1.0f/l;
		return RichVec4(v[0]*lm, v[1]*lm, v[2]*lm, v[3]*lm);
	}
	return *this;
}
void RichVec4::Lerp(const RichVec4 &vec, const float frac)
{
	v[0] = g_mfn->Math_LinearLerp(v[0], vec.v[0], frac);
	v[1] = g_mfn->Math_LinearLerp(v[1], vec.v[1], frac);
	v[2] = g_mfn->Math_LinearLerp(v[2], vec.v[2], frac);
	v[3] = g_mfn->Math_LinearLerp(v[3], vec.v[3], frac);
}
void RichVec4::Lerp(const RichVec4 &vecA, const RichVec4 &vecB, const float frac)
{
	v[0] = g_mfn->Math_LinearLerp(vecA.v[0], vecB.v[0], frac);
	v[1] = g_mfn->Math_LinearLerp(vecA.v[1], vecB.v[1], frac);
	v[2] = g_mfn->Math_LinearLerp(vecA.v[2], vecB.v[2], frac);
	v[3] = g_mfn->Math_LinearLerp(vecA.v[3], vecB.v[3], frac);
}

RichVec3 RichVec4::ToVec3(void) const
{
	return RichVec3(v[0], v[1], v[2]);
}
